<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use App\Services\PineconeService;
use Carbon\Carbon;
use Gemini\Laravel\Facades\Gemini;

class IndexJemaatData extends Command
{
    protected $signature = 'jemaat:index';
    protected $description = 'Indexes Jemaat data and generates embeddings';

    private $pineconeService;

    public function __construct(PineconeService $pineconeService)
    {
        $this->pineconeService = $pineconeService;
        parent::__construct();
    }
    
    public function handle()
    {
        $this->info('Starting Jemaat data indexing...');

        $this->pineconeService->deleteAllVectors(); #Delete all before doing a batch loading.

        $jemaats = DB::table('data_jemaat')->get();

        $batch = [];
        foreach ($jemaats as $jemaat) {
            // Calculate the age
            $today = Carbon::today();
            $birthdate = Carbon::parse($jemaat->TGL_LAHIR);
            $age = $birthdate->diffInYears($today);

            DB::table('data_jemaat')
                ->where('id', $jemaat->id)
                ->update(['umur' => $age]);


            $jemaatData = $this->createCombinedJemaatEmbedding($jemaat->id);
            if (!empty($jemaatData)) {
                $metadata = json_decode($jemaatData['metadata'], true); #Make sure the metadata is the right value

                $batch[] = [
                    'id' => 'jemaat-' . $jemaat->id,
                    'values' => json_decode($jemaatData['embedding'], true),
                    'metadata' => $metadata,
                ];
                $this->info("Jemaat " . $jemaat->NAMA . " has been embedded");
            } else {
                $this->info("Jemaat " . $jemaat->NAMA . " cannot be embedded");
            }
        }

        // Split batch into chunks of 5 and process each chunk
        $chunks = array_chunk($batch, 10);
        foreach ($chunks as $chunk) {
            $this->pineconeService->upsert($chunk); #Insert to PineconeDB
        }
        $this->info('Jemaat data indexed successfully.');
        return Command::SUCCESS;
    }

    // Move this to service for best maintainability
    function createCombinedJemaatEmbedding(int $jemaatId,string $stringEmpty = "Belum diisi", int $numberEmpty = -1): array
    {
        try{
            $jemaat = DB::table('data_jemaat')
                ->leftJoin('data_wilayah', 'data_jemaat.WILAYAH', '=', 'data_wilayah.id')
                ->leftJoin('data_sektor', 'data_jemaat.SEKTOR', '=', 'data_sektor.id')
                ->leftJoin('data_statuskl', 'data_jemaat.PERAN', '=', 'data_statuskl.id')
                ->leftJoin('data_kl', 'data_jemaat.ID_KELUARGA', '=', 'data_kl.ID')
                ->select([
                    "data_jemaat.id", # List it, so easy to track
                    "data_jemaat.ID_JEMAAT",
                    "data_jemaat.ID_SINODE",
                    "data_jemaat.ID_KELUARGA",
                    "data_jemaat.WILAYAH",
                    "data_jemaat.SEKTOR",
                    "data_jemaat.NAMA",
                    "data_jemaat.JK",
                    "data_jemaat.TEMPAT_KELAHIRAN",
                    "data_jemaat.TGL_LAHIR",
                    "data_jemaat.PERAN",
                    "data_jemaat.PERANETC",
                    "data_jemaat.ALAMAT",
                    "data_jemaat.KOTA",
                    "data_jemaat.PROPINSI",
                    "data_jemaat.KODEPOS",
                    "data_jemaat.NEGARA",
                    "data_jemaat.HP",
                    "data_jemaat.EMAIL",
                    "data_jemaat.STATUS",
                    "data_jemaat.KEWARGANEGARAAN",
                    "data_jemaat.GOL_DARAH",
                    "data_jemaat.STATUS_PERKAWINAN",
                    "data_jemaat.AYAH",
                    "data_jemaat.IBU",
                    "data_jemaat.TGL_BAPTIS",
                    "data_jemaat.TEMPAT_BAPTIS",
                    "data_jemaat.DIBAPTIS_OLEH",
                    "data_jemaat.TGL_SIDI",
                    "data_jemaat.TEMPAT_SIDI",
                    "data_jemaat.DISIDI_OLEH",
                    "data_jemaat.PINDAHAN",
                    "data_jemaat.PENDIDIKAN",
                    "data_jemaat.JURUSAN",
                    "data_jemaat.PEKERJAAN",
                    "data_jemaat.PEKERJAANETC",
                    "data_jemaat.TEMPAT_KERJA",
                    "data_jemaat.ALAMAT_KERJA",
                    "data_jemaat.BIDANG_USAHA",
                    "data_jemaat.JABATAN",
                    "data_jemaat.JABATANETC",
                    "data_jemaat.PROFESI",
                    "data_jemaat.PROFESIETC",
                    "data_jemaat.HOBBY",
                    "data_jemaat.HOBBYETC",
                    "data_jemaat.MINAT",
                    "data_jemaat.MINATETC",
                    "data_jemaat.MINAT_PELAYANAN",
                    "data_jemaat.updated_at",
                    "data_jemaat.updated_by",
                    "data_jemaat.TANGGAL_MENINGGAL",
                    "data_jemaat.TANGGAL_PINDAH",
                    "data_jemaat.TEMPAT_SEMAYAM",
                    "data_jemaat.DISABILITAS",
                    "data_jemaat.PINDAHAN_AGAMA_LAIN",
                    "data_jemaat.PINDAHAN_DARI_KRISTEN",
                    "data_jemaat.TGL_AGAMA_LAIN",
                    "data_jemaat.TGL_DARI_KRISTEN",
                    "data_wilayah.wilayah as nama_wilayah",
                    "data_sektor.nama as nama_sektor",
                    "data_statuskl.nama as peran_keluarga",
                    "data_kl.NAMA_KELUARGA as nama_keluarga"
                ])
                ->where('data_jemaat.ID_JEMAAT', $jemaatId)
                ->first();

            if (!$jemaat) {
                return [];
            }

            // 1. Concatenate Textual Data
            $text = $jemaat->NAMA . ". ";
            $text .= "Alamat: " . ($jemaat->ALAMAT ?? '') . ", " . ($jemaat->KOTA ?? '') . ". ";
            $text .= "Nama Keluarga: " . ($jemaat->nama_keluarga ?? $stringEmpty) . ". ";
            $text .= "Peran Keluarga: " . ($jemaat->peran_keluarga ?? $stringEmpty) . ". ";
            $text .= "Wilayah: " . ($jemaat->nama_wilayah ?? $stringEmpty) . ". ";
            $text .= "Sektor: " . ($jemaat->nama_sektor ?? $stringEmpty) . ". ";
            $text .= "Umur: " . $jemaat->umur . " Tahun. ";
            $text .= "Pendidikan: " . ($jemaat->PENDIDIKAN ?? $stringEmpty) . ". "; #Removed all the " "

            // 2. Generate Embedding
            $embedding = generateEmbeddings($text);

            // 3. Create Metadata
            $metadata = [
                'source' => 'jemaat',
                'id' => $jemaat->id,
                'wilayah' => $jemaat->WILAYAH ?? $numberEmpty, #Use number if no data
                'sektor' => $jemaat->SEKTOR ?? $numberEmpty, #Use number if no data
                'jk' => $jemaat->JK,
                'age' => $jemaat->umur,
            ];

            return [
                'text' => $text,
                'embedding' => json_encode($embedding),
                'metadata' => json_encode($metadata),
            ];
        }
        catch (\Exception $e) {
                \Log::error('Jemaat Indexing Error: ' . $e->getMessage() . ' - Jemaat ID: ' . $jemaatId); #Print message to see what is wrong
                return [];
            }
    }

    function generateEmbeddings($text, $endpoint_id = null, $project_id = null, $region = null) {
        $response = Gemini::embeddingModel('text-embedding-004')
            ->embedContent($text);
        
        return $response->embedding->values;
    }
}